const CACHE_KEY = "standorteNotes";
const CACHE_TIMESTAMP_KEY = "standorteNotesTimestamp";
const CACHE_DURATION = 60 * 60 * 1000; // 1 hour

const NOTES_URL = "https://project02.elias-wolfhard.dev/notes.json";

// ===============================
// Fetch Notes from Server
// ===============================
async function fetchAndCacheNotes() {
    const response = await fetch(NOTES_URL);
    if (!response.ok) {
        throw new Error("Failed to fetch notes");
    }

    const data = await response.json();
    const now = Date.now();

    await chrome.storage.local.set({
        [CACHE_KEY]: data,
        [CACHE_TIMESTAMP_KEY]: now
    });

    return data;
}

// ===============================
// Get Notes (Cache-first)
// ===============================
async function getNotes() {
    const result = await chrome.storage.local.get([
        CACHE_KEY,
        CACHE_TIMESTAMP_KEY
    ]);

    const now = Date.now();

    // Use cache if valid
    if (
        result[CACHE_KEY] &&
        result[CACHE_TIMESTAMP_KEY] &&
        now - result[CACHE_TIMESTAMP_KEY] < CACHE_DURATION
    ) {
        return result[CACHE_KEY];
    }

    return await fetchAndCacheNotes();
}

// ===============================
// Message Listener
// ===============================
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === "GET_NOTES") {
        getNotes()
            .then(data => {
                sendResponse({ success: true, data });
            })
            .catch(error => {
                sendResponse({ success: false, error: error.toString() });
            });

        return true; // Required for async response
    }
});
